<?php

namespace Modules\OpenAI\AiProviders\OpenAi\Traits;
use Illuminate\Http\Response;

trait OpenAiApiTrait
{
    private $audioUrl = 'https://api.openai.com/v1/audio/transcriptions';

    public function aiKey()
    {
        $apiKey = apiKey('openai');

        if (empty($apiKey)) {
            throw new \Exception(__("There's an issue with the :x API key. Kindly reach out to the administration for assistance.", ['x' => __('OpenAi')]), Response::HTTP_UNAUTHORIZED);
        }

        return $apiKey;
    }
 
    public function client()
    {
        return \OpenAI::client($this->aiKey());
    }

    public function chat()
    {
        return $this->client()->chat()->create($this->processedData);
    }

    public function chatStream()
    {
        return $this->client()->chat()->createStreamed($this->processedData);
    }

    public function speech()
    {
        return $this->client()->audio()->speech($this->processedData);
    }

    public function embeddings()
    {
        return $this->client()->embeddings()->create($this->processedData);
    }

    public function generate($headers, $url) 
    {
        $curl = curl_init();

        // Set cURL options
        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYHOST => config('openAI.ssl_verify_host'),
            CURLOPT_SSL_VERIFYPEER => config('openAI.ssl_verify_peer'),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => $this->processedData,
            CURLOPT_HTTPHEADER => array_merge($headers, [
                "Authorization: Bearer " . $this->aiKey(),
            ]),
        ));
        
        // Make API request
        $response = curl_exec($curl);
        $err = curl_error($curl);
        // Close cURL session
        curl_close($curl);

        $response = !empty($response) ? $response : $err;

        return json_decode($response, true);
    }

    public function audio()
    {
        return $this->client()->audio()->transcribe($this->processedData);
    }

    public function images()
    {
        return $this->client()->images()->create($this->processedData);
    }

    public function videoRequest(string $endpoint, string $method = 'POST', array $payload = [], bool $binary = false): array
    {
        $curl = curl_init();

        $url = rtrim(moduleConfig('openai.video_base_url'), '/') . '/' . ltrim($endpoint, '/');

        $timeout = $binary ? 120 : 60; // Higher timeout for video operations

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYHOST => config('openAI.ssl_verify_host'),
            CURLOPT_SSL_VERIFYPEER => config('openAI.ssl_verify_peer'),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => strtoupper($method),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . $this->aiKey(),
            ],
        ];

        if ($binary) {
            // For binary content (video download), don't set Content-Type
            $options[CURLOPT_FOLLOWLOCATION] = true;
        } elseif (strtoupper($method) === 'POST' && !empty($payload)) {
            // Check if payload contains a file (for multipart/form-data)
            $hasFile = false;
            foreach ($payload as $key => $value) {
                if ($value instanceof \Illuminate\Http\UploadedFile || is_resource($value)) {
                    $hasFile = true;
                    break;
                }
            }

            if ($hasFile) {
                // Use multipart/form-data for file uploads
                $postData = [];
                foreach ($payload as $key => $value) {
                    if ($value instanceof \Illuminate\Http\UploadedFile) {
                        $postData[$key] = new \CURLFile(
                            $value->getRealPath(),
                            $value->getMimeType(),
                            $value->getClientOriginalName()
                        );
                    } else {
                        $postData[$key] = $value;
                    }
                }
                $options[CURLOPT_POSTFIELDS] = $postData;
            } else {
                // Use JSON for regular requests
                $options[CURLOPT_HTTPHEADER][] = "Content-Type: application/json";
                $options[CURLOPT_POSTFIELDS] = json_encode($payload);
            }
        }

        curl_setopt_array($curl, $options);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        curl_close($curl);

        if ($response === false) {
            return [
                'code' => 0,
                'body' => ['error' => ['message' => $error ?: 'Connection failed']],
            ];
        }

        if ($binary) {
            // Return binary data directly
            return [
                'code' => $statusCode,
                'body' => $response, // Binary video data
                'binary' => true,
            ];
        }

        return [
            'code' => $statusCode,
            'body' => json_decode($response, true),
        ];
    }

}
    
