<?php

namespace Modules\OpenAI\AiProviders\OpenAi\Responses\Traits;

use Exception;
use Illuminate\Support\Facades\Log;

trait FetchVideoTrait
{
    /**
     * Process the video fetch response
     *
     * @return mixed
     */
    public function processVideoFetch(): mixed
    {
        if ($this->response['code'] === 401) {
            $this->handleException(__("There's an issue with the :x API key. Kindly reach out to the administration for assistance.", ['x' => 'OpenAI']));
        }

        if (!in_array($this->response['code'], [200, 202], true)) {
            $body = $this->response['body'] ?? [];

            // If body is a JSON string, decode it
            if (is_string($body)) {
                $decodedBody = json_decode($body, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $body = $decodedBody;
                }
            }

            $message = data_get($body, 'error.message') ?: __('Something went wrong, please try again.');
            $this->handleException($message);
        }

        // OpenAI video API /content endpoint returns binary video data directly
        if (!empty($this->response['binary'])) {
            $this->addVideoContent($this->response['body']);
        } else {
            // Fallback: try to extract URL if response is JSON
            $videoUrl = data_get($this->response, 'body.output[0].url');
            if ($videoUrl) {
                $this->fetchVideoFromUrl($videoUrl);
            } else {
                // No binary data and no URL found
                $this->handleException(__('Video content not found in the response.'));
            }
        }

        $videoContent = $this->getVideoContent();
        
        // Ensure we actually have video content
        if (empty($videoContent)) {
            Log::error('FetchVideoTrait: Video content is empty after processing', [
                'response_code' => $this->response['code'] ?? null
            ]);
            $this->handleException(__('Video content is empty or could not be processed.'));
        }

        return $videoContent;
    }

    /**
     * Fetch video content from URL
     *
     * @param string $videoUrl
     * @return void
     */
    protected function fetchVideoFromUrl(string $videoUrl): void
    {
        $context = stream_context_create([
            'http' => [
                'timeout' => 30,
                'ignore_errors' => true,
            ]
        ]);

        $videoContent = @file_get_contents($videoUrl, false, $context);
        
        if ($videoContent === false) {
            $error = error_get_last();
            $errorMessage = $error['message'] ?? __('Failed to fetch video from URL.');
            $this->handleException(__('Failed to download video: :message', ['message' => $errorMessage]));
        }

        $this->addVideoContent($videoContent);
    }

    /**
     * Add video content to the collection
     *
     * @param mixed $content
     * @return void
     */
    abstract protected function addVideoContent(mixed $content): void;

    /**
     * Get the video content collection
     *
     * @return array
     */
    abstract protected function getVideoContent(): array;

    /**
     * Get the response content
     *
     * @return mixed
     */
    public function response(): mixed
    {
        return data_get($this->response, 'body', []);
    }

    /**
     * Handle exceptions
     *
     * @param string $message
     * @return Exception
     */
    public function handleException(string $message): Exception
    {
        throw new Exception($message);
    }
}